/*
**  OSSP l2 - Flexible Logging
**  Copyright (c) 2001-2003 Cable & Wireless Deutschland GmbH
**  Copyright (c) 2001-2003 The OSSP Project (http://www.ossp.org/)
**  Copyright (c) 2001-2003 Ralf S. Engelschall <rse@engelschall.com>
**
**  This file is part of OSSP l2, a flexible logging library which
**  can be found at http://www.ossp.org/pkg/lib/l2/.
**
**  Permission to use, copy, modify, and distribute this software for
**  any purpose with or without fee is hereby granted, provided that
**  the above copyright notice and this permission notice appear in all
**  copies.
**
**  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
**  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
**  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
**  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
**  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
**  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
**  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
**  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
**  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
**  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
**  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
**  SUCH DAMAGE.
**
**  l2_ch_file.c: file channel implementation
*/

#include "l2.h"

#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>

/* declare private channel configuration */
typedef struct {
    int   fd;
    char *path;
    int   append;
    int   trunc;
    int   perm;
    int   jitter;
} l2_ch_file_t;

/* create channel */
static l2_result_t hook_create(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_file_t *cfg;

    /* allocate private channel configuration */
    if ((cfg = (l2_ch_file_t *)malloc(sizeof(l2_ch_file_t))) == NULL)
        return L2_ERR_ARG;

    /* initialize configuration with reasonable defaults */
    cfg->fd     = -1;
    cfg->path   = NULL;
    cfg->append = -1;
    cfg->trunc  = -1;
    cfg->perm   = (S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
    cfg->jitter = 0;

    /* link private channel configuration into channel context */
    ctx->vp = cfg;

    return L2_OK;
}

/* configure channel */
static l2_result_t hook_configure(l2_context_t *ctx, l2_channel_t *ch, const char *fmt, va_list ap)
{
    l2_ch_file_t *cfg = (l2_ch_file_t *)ctx->vp;
    l2_param_t pa[6];
    l2_result_t rv;
    l2_env_t *env;

    /* feed and call generic parameter parsing engine */
    L2_PARAM_SET(pa[0], path,   STR, &cfg->path);
    L2_PARAM_SET(pa[1], append, INT, &cfg->append);
    L2_PARAM_SET(pa[2], trunc,  INT, &cfg->trunc);
    L2_PARAM_SET(pa[3], perm,   INT, &cfg->perm);
    L2_PARAM_SET(pa[4], jitter, INT, &cfg->jitter);
    L2_PARAM_END(pa[5]);
    l2_channel_env(ch, &env);
    rv = l2_util_setparams(env, pa, fmt, ap);

    return rv;
}

/* open channel */
static l2_result_t hook_open(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_file_t *cfg = (l2_ch_file_t *)ctx->vp;
    int mode;
    mode_t mask;

    /* "append" backward compatibility; only cfg->trunc is used in the code
     * make sure append/trunc either both use defaults, both are set different, or only one is set
     *
     * truth table for user input,       append, trunc => resulting trunc
     *                 -----------------+------+------+------------------
     *                                       -1     -1      0 (default)
     *                 trunc=0               -1      0      0 
     *                 trunc=1               -1      1      1 
     *                 append=0               0     -1      1 
     *                 append=0, trunc=0      0      0      ERROR
     *                 append=0, trunc=1      0      1      1 
     *                 append=1               1     -1      0 
     *                 append=1, trunc=0      1      0      0 
     *                 append=1, trunc=1      1      1      ERROR
     */
    if (cfg->append >= 1)
        cfg->append = 1; /* reduce to -1 (undef), 0 (no), 1 (yes) */
    if (cfg->trunc >= 1)
        cfg->trunc = 1; /* reduce to -1 (undef), 0 (no), 1 (yes) */
    if (   cfg->append != -1
        && cfg->trunc != -1
        && cfg->append == cfg->trunc) /* collision */
        return L2_ERR_USE;
    if (   cfg->trunc == -1)
        cfg->trunc = (1 - cfg->append) & 1;

    /* reduce jitter to 0 (no) or 1 (yes) */
    if (cfg->jitter <= 0)
        cfg->jitter = 0;
    if (cfg->jitter >= 1)
        cfg->jitter = 1;

    /* make sure a path was set */
    if (cfg->path == NULL)
        return L2_ERR_USE;

    /* open channel file */
    mode = O_WRONLY|O_CREAT;
    if (cfg->trunc == 1)
        mode |= O_TRUNC;
    else
        mode |= O_APPEND;
    mask = umask(0);
    cfg->fd = open(cfg->path, mode, cfg->perm);
    umask(mask);
    if (cfg->fd == -1)
        return L2_ERR_SYS;

    /* close channel file if jittering, we just wanted to see a successful open and truncate if required */
    if (cfg->jitter == 1) {
        close(cfg->fd);
        cfg->fd = -1;
    }

    return L2_OK;
}

/* write to channel */
static l2_result_t hook_write(l2_context_t *ctx, l2_channel_t *ch,
                              l2_level_t level, const char *buf, size_t buf_size)
{
    l2_ch_file_t *cfg = (l2_ch_file_t *)ctx->vp;
    l2_result_t rc = L2_OK;
    mode_t mask;

    /* handle the impossible; close any lingering file if jittering */
    if (cfg->fd != -1 && cfg->jitter == 1) {
        close(cfg->fd);
        cfg->fd = -1;
    }

    /* open channel file if jittering */
    if (cfg->fd == -1 && cfg->jitter == 1) {
        mask = umask(0);
        cfg->fd = open(cfg->path, O_WRONLY|O_CREAT|O_APPEND, cfg->perm);
        umask(mask);
    }
    if (cfg->fd == -1)
        return L2_ERR_SYS;

    /* write message to channel file */
    if (write(cfg->fd, buf, buf_size) == -1)
        rc = L2_ERR_SYS;

    /* close channel file if jittering */
    if (cfg->jitter == 1) {
        close(cfg->fd);
        cfg->fd = -1;
    }

    return rc;
}

/* close channel */
static l2_result_t hook_close(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_file_t *cfg = (l2_ch_file_t *)ctx->vp;

    /* close channel file */
    close(cfg->fd);
    cfg->fd = -1;

    return L2_OK;
}

/* destroy channel */
static l2_result_t hook_destroy(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_file_t *cfg = (l2_ch_file_t *)ctx->vp;

    /* destroy channel configuration */
    if (cfg->path != NULL)
        free(cfg->path);
    free(cfg);

    return L2_OK;
}

/* exported channel handler structure */
l2_handler_t l2_handler_file = {
    "file",
    L2_CHANNEL_OUTPUT,
    hook_create,
    hook_configure,
    hook_open,
    hook_write,
    NULL,
    hook_close,
    hook_destroy
};

