/*
**  OSSP sio - Stream I/O
**  Copyright (c) 2002-2003 Cable & Wireless Deutschland <http://www.cw.com/de/>
**  Copyright (c) 2002-2003 The OSSP Project <http://www.ossp.org/>
**  Copyright (c) 2002-2003 Ralf S. Engelschall <rse@engelschall.com>
**
**  This file is part of OSSP sio, a layered stream I/O library
**  which can be found at http://www.ossp.org/pkg/lib/sio/.
**
**  Permission to use, copy, modify, and distribute this software for
**  any purpose with or without fee is hereby granted, provided that
**  the above copyright notice and this permission notice appear in all
**  copies.
**
**  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
**  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
**  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
**  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
**  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
**  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
**  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
**  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
**  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
**  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
**  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
**  SUCH DAMAGE.
**
**  sio_test.c: test suite
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>

#include "al.h"
#include "sio.h"

#include "sa.h"

#include <openssl/ssl.h>
#include <openssl/bio.h>
extern BIO_METHOD *BIO_s_socket();

extern sio_module_t sio_module_bio;
extern sio_module_t sio_module_hello;
extern sio_module_t sio_module_buffer;
#ifndef SINK
extern sio_module_t sio_module_sa;
#endif

#define e(f) rc = f; printf("%s = %s\n",#f, sio_error(rc)); fflush(stdout);
#define s(f) src = f; printf("%s = %s\n",#f, sa_error(src)); fflush(stdout);
#define n(f) rn = f; printf("%s = %d\n",#f, rn); fflush(stdout);
#define p(f) rp = f; printf("%s = %p\n",#f, rp); fflush(stdout);

int main(int argc, char *argv[])
{
    int rn;
    void *rp;
    sio_rc_t rc;
    sio_t *sio;
#ifndef SINK
    sio_stage_t *sios_sa;
#endif
    sio_stage_t *sios_bio, *sios_hello, *sios_buffer;

    sa_rc_t src;
    sa_addr_t *saa;
    sa_t *msa, *sa;
    char *uri;

    int fd;
    SSL_CTX *ctx;
#ifdef SINK
    BIO *bio;
#endif
    BIO *sbio;

    char buf[] = "Hello world\n";

    size_t actual;
    size_t buflen;
    int no  = 0;
    int yes = 1;

    s(sa_create(&msa));
    s(sa_option(msa, SA_OPTION_REUSEADDR, 1));

    s(sa_addr_create(&saa));
    uri = "inet://localhost:25001#tcp";
    s(sa_addr_u2a(saa, uri));
    s(sa_bind(msa,saa));
    s(sa_addr_destroy(saa));
    uri = NULL;

    ERR_load_BIO_strings();
    OpenSSL_add_ssl_algorithms();
    ctx = SSL_CTX_new(SSLv23_server_method());

    if (!SSL_CTX_use_certificate_file(ctx, "/u/mlelstv/ssl/server.crt", SSL_FILETYPE_PEM))
        exit(98);
    if (!SSL_CTX_use_PrivateKey_file(ctx, "/u/mlelstv/ssl/server.key", SSL_FILETYPE_PEM))
        exit(99);

    e(sio_create(&sio));
    e(sio_create_stage(sio, &sio_module_bio,    &sios_bio));
    e(sio_create_stage(sio, &sio_module_hello,  &sios_hello));
    e(sio_create_stage(sio, &sio_module_buffer, &sios_buffer));
#ifndef SINK
    e(sio_create_stage(sio, &sio_module_sa,     &sios_sa));
#endif

    buflen = 256;
    e(sio_configure_stage(sio, sios_bio, "inputsize", &buflen));

    buflen = 1000;
    e(sio_configure_stage(sio, sios_buffer, "outputsize", &buflen));

    s(sa_listen(msa, 5));

    for(;;) {

        s(sa_accept(msa, &saa, &sa));
        s(sa_addr_a2u(saa, &uri));
        printf("Connection from %s\n",uri);
        s(sa_addr_destroy(saa));

        p(sbio = BIO_new_ssl(ctx,0));
        e(sio_configure_stage(sio, sios_bio, "bio", sbio));

#ifdef SINK
        s(sa_getfd(sa, &fd));
        p(bio = BIO_new_socket(fd, 0));
        p(BIO_push(sbio,bio));
#else
        e(sio_configure_stage(sio, sios_sa, "sa", sa));
        buflen = 256;
        e(sio_configure_stage(sio, sios_sa, "buflen", &buflen));
        e(sio_configure_stage(sio, sios_bio, "issink", &no));
#endif
        e(sio_configure_stage(sio, sios_bio, "freebio", &yes));


#ifndef SINK
        e(sio_attach(sio, sios_sa, SIO_MODE_READWRITE));
#endif
        e(sio_attach(sio, sios_bio, SIO_MODE_READWRITE));
        e(sio_attach(sio, sios_hello, SIO_MODE_READWRITE));
        e(sio_attach(sio, sios_buffer, SIO_MODE_WRITE));

        e(sio_write(sio, buf, sizeof(buf)-1, &actual));
        e(sio_push(sio));

        e(sio_detach(sio, sios_buffer));
        e(sio_detach(sio, sios_hello));
        e(sio_detach(sio, sios_bio));
#ifndef SINK
        e(sio_detach(sio, sios_sa));
#endif

#ifdef SINK
        BIO_pop(bio);
#endif
        /* BIO_free(sbio); */
#ifdef SINK
        BIO_free(bio);
#endif

        sa_destroy(sa);
    }


#ifndef SINK
    e(sio_destroy_stage(sio, sios_sa));
#endif
    e(sio_destroy_stage(sio, sios_buffer));
    e(sio_destroy_stage(sio, sios_hello));
    e(sio_destroy_stage(sio, sios_bio));

    e(sio_destroy(sio));

    sa_destroy(msa);

    return 0;
}
