/*
 * as_uuidgen.c --- generate a DCE-compatible uuid
 *
 * Copyright (C) 1996, 1997, 1998, 1999 Theodore Ts'o.
 *
 * This file may be redistributed under the terms of the GNU 
 * Library General Public License.
 */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include "as_uuidgen.h"

/* Include whatever autoconf found */
#ifdef HAVE_CONFIG_H
#include "ac_config.h"
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_SYS_SOCKIO_H
#include <sys/sockio.h>
#endif /* HAVE_SYS_SOCKIO_H */
#ifdef HAVE_NET_IF_H
#include <net/if.h>
#endif /* HAVE_NET_IF_H */
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif /* HAVE_NETINET_IN_H */

#ifndef max
/* Define a general use max macro */
#define max(a,b) ((a) > (b) ? (a) : (b))
#endif /* max */

#ifdef HAVE_NET_IF_H
/* Define the size of a interface address */
#define ifreq_size(i) max(sizeof(struct ifreq), sizeof((i).ifr_name)+(i).ifr_addr.sa_len)
#endif /* HAVE_NET_IF_H */


static int get_random_fd(void)
{
    struct timeval  tv;
    static int      fd = -2;
    int             i;

    if (fd == -2) {
        gettimeofday(&tv, 0);
        fd = open("/dev/urandom", O_RDONLY);
        if (fd == -1)
            fd = open("/dev/random", O_RDONLY | O_NONBLOCK);
        srand((getpid() << 16) ^ getuid() ^ tv.tv_sec ^ tv.tv_usec);
    }
    /* Crank the random number generator a few times */
    gettimeofday(&tv, 0);
    for (i = (tv.tv_sec ^ tv.tv_usec) & 0x1F; i > 0; i--)
        rand();
    return fd;
}


/*
 * Generate a series of random bytes.  Use /dev/urandom if possible,
 * and if not, use srandom/random.
 */
static void get_random_bytes(void *buf, int nbytes)
{
    int i, fd = get_random_fd();
    int lose_counter = 0;
    char *cp = (char *) buf;

    if (fd >= 0) {
        while (nbytes > 0) {
            i = read(fd, cp, nbytes);
            if ((i < 0) &&
                ((errno == EINTR) || (errno == EAGAIN)))
                continue;
            if (i <= 0) {
                if (lose_counter++ == 8)
                    break;
                continue;
            }
            nbytes -= i;
            cp += i;
            lose_counter = 0;
        }
    }

    /* XXX put something better here if no /dev/random! */
    for (i = 0; i < nbytes; i++)
        *cp++ = rand() & 0xFF;
    return;
}

/*
 * Get the ethernet hardware address, if we can find it...
 */
static int get_node_id(unsigned char *node_id)
{
#ifdef HAVE_NET_IF_H
    int             sd;
    struct ifreq    ifr, *ifrp;
    struct ifconf   ifc;
    char            buf[1024];
    int             n, i;
    unsigned char   *a;
    
    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
    if (sd < 0) {
        return -1;
    }
    memset(buf, 0, sizeof(buf));
    ifc.ifc_len = sizeof(buf);
    ifc.ifc_buf = buf;
    if (ioctl (sd, SIOCGIFCONF, (char *)&ifc) < 0) {
        close(sd);
        return -1;
    }
    n = ifc.ifc_len;
    for (i = 0; i < n; i+= ifreq_size(ifr)) {
        ifrp = (struct ifreq *)((char *) ifc.ifc_buf+i);
        strncpy(ifr.ifr_name, ifrp->ifr_name, IFNAMSIZ);
#ifdef HAVE_SIOCGIFHWADDR
        if (ioctl(sd, SIOCGIFHWADDR, &ifr) < 0)
            continue;
        a = (unsigned char *) &ifr.ifr_hwaddr.sa_data;
#else
#ifdef HAVE_SIOCGENADDR
        if (ioctl(sd, SIOCGENADDR, &ifr) < 0)
            continue;
        a = (unsigned char *) ifr.ifr_enaddr;
#else
        /*
         * Broken! We don't have a way of getting the hardware
         * address, so just return without doing anything else
         */
        close(sd);
        return 0;
#endif /* SIOCGENADDR */
#endif /* SIOCGIFHWADDR */
        if (!a[0] && !a[1] && !a[2] && !a[3] && !a[4] && !a[5])
            continue;
        if (node_id) {
            memcpy(node_id, a, 6);
            close(sd);
            return 1;
        }
    }
    close(sd);
#endif
    return 0;
}

/* Assume that the gettimeofday() has microsecond granularity */
#define MAX_ADJUSTMENT 10

static int get_clock(__u32 *clock_high, __u32 *clock_low, __u16 *ret_clock_seq)
{
    static int              adjustment = 0;
    static struct timeval   last = {0, 0};
    static __u16            clock_seq;
    struct timeval          tv;
    unsigned long long      clock_reg;
    
try_again:
    gettimeofday(&tv, 0);
    if ((last.tv_sec == 0) && (last.tv_usec == 0)) {
        get_random_bytes(&clock_seq, sizeof(clock_seq));
        clock_seq &= 0x1FFF;
        last = tv;
        last.tv_sec--;
    }
    if ((tv.tv_sec < last.tv_sec) ||
        ((tv.tv_sec == last.tv_sec) &&
         (tv.tv_usec < last.tv_usec))) {
        clock_seq = (clock_seq+1) & 0x1FFF;
        adjustment = 0;
        last = tv;
    } else if ((tv.tv_sec == last.tv_sec) &&
        (tv.tv_usec == last.tv_usec)) {
        if (adjustment >= MAX_ADJUSTMENT)
            goto try_again;
        adjustment++;
    } else {
        adjustment = 0;
        last = tv;
    }

    clock_reg = tv.tv_usec*10 + adjustment;
    clock_reg += ((unsigned long long) tv.tv_sec)*10000000;
    clock_reg += (((unsigned long long) 0x01B21DD2) << 32) + 0x13814000;

    *clock_high = clock_reg >> 32;
    *clock_low = clock_reg;
    *ret_clock_seq = clock_seq;
    return 0;
}

void uuid_generate_time(uuid_t out)
{
    static unsigned char    node_id[6];
    static int              has_init = 0;
    struct uuid             uu;
    __u32                   clock_mid;

    if (!has_init) {
        if (get_node_id(node_id) <= 0) {
            get_random_bytes(node_id, 6);
            /*
             * Set multicast bit, to prevent conflicts
             * with IEEE 802 addresses obtained from
             * network cards
             */
            node_id[0] |= 0x80;
        }
        has_init = 1;
    }
    get_clock(&clock_mid, &uu.time_low, &uu.clock_seq);
    uu.clock_seq |= 0x8000;
    uu.time_mid = (__u16) clock_mid;
    uu.time_hi_and_version = (clock_mid >> 16) | 0x1000;
    memcpy(uu.node, node_id, 6);
    uuid_pack(&uu, out);
}

void uuid_generate_random(uuid_t out)
{
    uuid_t      buf;
    struct uuid uu;

    get_random_bytes(buf, sizeof(buf));
    uuid_unpack(buf, &uu);

    uu.clock_seq = (uu.clock_seq & 0x3FFF) | 0x8000;
    uu.time_hi_and_version = (uu.time_hi_and_version & 0x0FFF) | 0x4000;
    uuid_pack(&uu, out);
}

/*
 * This is the generic front-end to uuid_generate_random and
 * uuid_generate_time.  It uses uuid_generate_random only if
 * /dev/urandom is available, since otherwise we won't have
 * high-quality randomness.
 */
void uuid_generate(uuid_t out)
{
    if (get_random_fd() >= 0)
        uuid_generate_random(out);
    else
        uuid_generate_time(out);
}

void uuid_pack(const struct uuid *uu, uuid_t ptr)
{
    __u32           tmp;
    unsigned char   *out = ptr;

    tmp = uu->time_low;
    out[3] = (unsigned char) tmp;
    tmp >>= 8;
    out[2] = (unsigned char) tmp;
    tmp >>= 8;
    out[1] = (unsigned char) tmp;
    tmp >>= 8;
    out[0] = (unsigned char) tmp;

    tmp = uu->time_mid;
    out[5] = (unsigned char) tmp;
    tmp >>= 8;
    out[4] = (unsigned char) tmp;

    tmp = uu->time_hi_and_version;
    out[7] = (unsigned char) tmp;
    tmp >>= 8;
    out[6] = (unsigned char) tmp;

    tmp = uu->clock_seq;
    out[9] = (unsigned char) tmp;
    tmp >>= 8;
    out[8] = (unsigned char) tmp;

    memcpy(out+10, uu->node, 6);
}

void uuid_unpack(const uuid_t in, struct uuid *uu)
{
    const __u8   *ptr = in;
    __u32        tmp;

    tmp = *ptr++;
    tmp = (tmp << 8) | *ptr++;
    tmp = (tmp << 8) | *ptr++;
    tmp = (tmp << 8) | *ptr++;
    uu->time_low = tmp;

    tmp = *ptr++;
    tmp = (tmp << 8) | *ptr++;
    uu->time_mid = tmp;

    tmp = *ptr++;
    tmp = (tmp << 8) | *ptr++;
    uu->time_hi_and_version = tmp;

    tmp = *ptr++;
    tmp = (tmp << 8) | *ptr++;
    uu->clock_seq = tmp;

    memcpy(uu->node, ptr, 6);
}

void uuid_unparse(const uuid_t uu, char *out)
{
    struct uuid uuid;

    uuid_unpack(uu, &uuid);
    sprintf(out,
        "%08x-%04x-%04x-%02x%02x-%02x%02x%02x%02x%02x%02x",
        uuid.time_low, uuid.time_mid, uuid.time_hi_and_version,
        uuid.clock_seq >> 8, uuid.clock_seq & 0xFF,
        uuid.node[0], uuid.node[1], uuid.node[2],
        uuid.node[3], uuid.node[4], uuid.node[5]);
}
