//
//  OSSP asgui - Accounting system graphical user interface
//  Copyright (c) 2002-2003 The OSSP Project (http://www.ossp.org/)
//  Copyright (c) 2002-2003 Cable & Wireless Deutschland (http://www.cw.com/de/)
//  Copyright (c) 2002-2003 Ralf S. Engelschall <rse@engelschall.com>
//  Copyright (c) 2002-2003 Michael Schloh von Bennewitz <michael@schloh.com>
//
//  This file is part of OSSP asgui, an accounting system graphical user
//  interface which can be found at http://www.ossp.org/pkg/tool/asgui/.
//
//  Permission to use, copy, modify, and distribute this software for
//  any purpose with or without fee is hereby granted, provided that
//  the above copyright notice and this permission notice appear in all
//  copies.
//
//  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
//  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
//  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
//  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
//  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
//  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
//  SUCH DAMAGE.
//
//  titslot.cpp: ISO C++ implementation
//

// Qt headers
#include <qfiledialog.h>
#include <qcombobox.h>
#include <qregexp.h>
#include <qclipboard.h>

// User interface
#include "as_gui.h"             // Main classes
#include "as_except.h"          // Exception classes
#include "as_tableitem.h"       // For our custom table items
#include "as_generic.h"         // Generic classes

// Icon pixel maps
#include "as_gfx/cwlogo.xpm"    // static const char *s_kpcCwlogo_xpm[]
#include "as_gfx/ossplogo.xpm"  // static const char *s_kpcOssplogo_xpm[]


//
// Cut an entry
//
void Titraqform::cutEntry(void)
{
    this->copyEntry();  // Reuse slot
    this->delEntry();   // Reuse slot
}

//
// Copy an entry
//
void Titraqform::copyEntry(void)
{
    QString Selection;  // Will hold the selected text
    QClipboard *pClip;  // Will reference the global clipboard

    // Initialize data and clipboard handle
    Selection = getRowdata();   // Use accessor
    pClip = QApplication::clipboard();

    Q_ASSERT(!Selection.isNull());
    pClip->setText(Selection, QClipboard::Selection); // Doesn't work on Windows
    pClip->setText(Selection, QClipboard::Clipboard); // Works on both equally
}

//
// Paste an entry
//
void Titraqform::pasteEntry(void)
{
    int nRows = 0;              // Paste so many rows as are stored
    QString Selection;          // Will receive the clipboard text
    QClipboard *pClip;          // Will reference the global clipboard

    Selection = pClip->text(QClipboard::Clipboard); // Windows and Unix
    nRows = Selection.contains(QChar('\n'));        // How many rows
    this->addEntry(nRows);                          // Reuse slot

    if (Selection) {
        setRowdata(Selection);                      // Use accessor
        this->setDirty(true);                       // Set dirty state
    }
}

//
// Append a blank row entry
//
void Titraqform::addEntry(int nRows)
{
    QTableSelection Select; // Highlighted text
    int nTotal = 0;         // Total row select
    int nCurrent = 0;       // Current row

    if (nRows == -1) {      // Assume a user selection range
        Select = m_pMaintable->selection(0);
        nTotal = Select.bottomRow() - Select.topRow() + 1;
        // Add a row after selection and focus to the new row
        if (Select.bottomRow() + 1 != m_pMaintable->numRows()) { // Add upwards
            m_pMaintable->insertRows(Select.topRow(), nTotal);
            this->setDirty(true);   // Set data to dirty state
            m_pMaintable->setCurrentCell(Select.topRow(), m_pMaintable->currentColumn());
        }
        else { // Special case on last row add downwards
            m_pMaintable->insertRows(Select.bottomRow() + 1, nTotal);
            this->setDirty(true);   // Set data to dirty state
            m_pMaintable->setCurrentCell(Select.bottomRow() + 1, m_pMaintable->currentColumn());
            m_pMaintable->ensureCellVisible(m_pMaintable->numRows() - 1, 0); // Scroll please
        }
    }
    else {                  // Do not count a user selection
        nTotal   = nRows;
        nCurrent = m_pMaintable->currentRow();
        // Add a row after selection and focus to the new row
        if (nCurrent + 1 != m_pMaintable->numRows()) { // Add upwards
            m_pMaintable->insertRows(nCurrent, nTotal);
            this->setDirty(false);  // Reset data to clean state
            m_pMaintable->setCurrentCell(nCurrent, m_pMaintable->currentColumn());
        }
        else { // Special case on last row add downwards
            m_pMaintable->insertRows(nCurrent + 1, nTotal);
            this->setDirty(false);  // Reset data to clean state
            m_pMaintable->setCurrentCell(nCurrent + 1, m_pMaintable->currentColumn());
            m_pMaintable->ensureCellVisible(m_pMaintable->numRows() - 1, 0); // Scroll please
        }
    }

//    updateDate(*m_pDatezero);
//    updateStart(QTime::QTime(0, 0));
//    updateFinish(QTime::QTime(0, 0));
//    updateAmount(trUtf8("00:00"));

//    char szGuitext[37];
//    uuid_t Guid;
//
//    uuid_generate(Guid);
//    uuid_unparse(Guid, szGuitext);
//    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXGUID, szGuitext);
}

//
// Delete a row entry
//
void Titraqform::delEntry(int nRows)
{
    QTableSelection Select = m_pMaintable->selection(0);    // Highlighted text
    int nTotal = Select.bottomRow() - Select.topRow() + 1;  // Total row select
    QMemArray<int> Rowselect(nTotal);                       // Row array

    // Calculate rows to delete from selection highlight
    for (int nIter = 0; nIter < nTotal; ++nIter)
        Rowselect[nIter] = Select.topRow() + nIter;

    // Remove the row at selection and focus to the next row
    if (m_pMaintable->currentRow() + 1 != m_pMaintable->numRows()) {
        m_pMaintable->setCurrentCell(Select.bottomRow() + 1, m_pMaintable->currentColumn());
        m_pMaintable->removeRows(Rowselect);
        this->setDirty(true);   // Set data to dirty state
    }
    else {  // Special case to handle removing of only row or last row
        m_pMaintable->removeRows(Rowselect);
        this->setDirty(true);   // Set data to dirty state
    }
}

//
// Make and display a new document window
//
void Titraqform::newDoc(void)
{
    int nResult = 0;

    // Check modification state of current data
    if (this->isDirty()) {
        nResult = QMessageBox::information(this, trUtf8(TITRAQ_APPTITLE),
                  trUtf8(TITRAQ_SAVEFIRST), trUtf8("&Save"),
                  trUtf8("&Discard"), trUtf8("Cancel"), 0, 2);

        switch (nResult) {
        case 0: // Save first
            this->saveFile();               // Save changes first
            m_pMaintable->setNumRows(0);    // Remove all data in table
            this->setDirty(false);          // Reset data to clean state
            break;
        case 1: // Don't save first but do new
            m_pMaintable->setNumRows(0);    // Remove all data in table
            this->setDirty(false);          // Reset data to clean state
            break;
        case 2: // Don't do a new timesheet
        default:
            break;
        }
    }
    else {                                  // Data is already up to date
        m_pMaintable->setNumRows(0);        // Remove all data in table
        this->setDirty(false);              // Reset data to clean state
    }
}

//
// Open and display an existing document
//
void Titraqform::openDoc(void)
{
    int nResult = 0;    // Holds return value from save first messagebox
    QString Filestring = QFileDialog::getOpenFileName("/e/dev/as", QString::null, this, trUtf8("Chooser Dialog"), trUtf8("Choose a file to open"));

    if (!Filestring.isEmpty()) {
        QFile Filetemp(Filestring); // File to load
        // Check modification state of current data
        if (this->isDirty()) {
            nResult = QMessageBox::information(this, trUtf8(TITRAQ_APPTITLE),
                      trUtf8(TITRAQ_SAVEFIRST), trUtf8("&Save"),
                      trUtf8("&Discard"), trUtf8("Cancel"), 0, 2);

            switch (nResult) {
            case 0: // Save first
                this->saveFile();               // Save changes first
                m_szFilename->operator=(Filestring);
                m_pMaintable->setNumRows(0);    // Clear out old data
                try {
                    loadData(Filetemp);         // Pass to helper method
                }
                catch (Genexcept& Genex) {
                    Genex.reportErr();
                }
                // Reset and give output to main window
                this->setCaption(Filestring);
                m_pStatbar->message(trUtf8("Loaded document ") + Filestring, 4000);
                this->setDirty(false);          // Set the clean state
                break;
            case 1: // Don't save first but do load
                m_szFilename->operator=(Filestring);
                m_pMaintable->setNumRows(0);    // Clear out old data
                try {
                    loadData(Filetemp);         // Pass to helper method
                }
                catch (Genexcept& Genex) {
                    Genex.reportErr();
                }
                // Reset and give output to main window
                this->setCaption(Filestring);
                m_pStatbar->message(trUtf8("Loaded document ") + Filestring, 4000);
                this->setDirty(false);          // Set the clean state
                break;
            case 2: // Don't do a load timesheet
            default:
                break;
            }
        }
        else {
            m_szFilename->operator=(Filestring);
            m_pMaintable->setNumRows(0);    // Clear out old data
            try {
                loadData(Filetemp);         // Pass to helper method
            }
            catch (Genexcept& Genex) {
                Genex.reportErr();
            }
            // Reset and give output to main window
            this->setCaption(Filestring);
            m_pStatbar->message(trUtf8("Loaded document ") + Filestring, 4000);
            this->setDirty(false);          // Set the clean state
        }
    }
    else
        m_pStatbar->message(trUtf8("Loading aborted"), 4000);
}

//
// Serialize current state to the current file
//
void Titraqform::saveFile(void)
{
    // First time saves are really just saveAs in disguise
    if (m_szFilename->isEmpty()) {
        saveAs();
        return;
    }

    // Try to open a file for writing to
    QFile Filetemp(*m_szFilename);
    try {
        saveData(Filetemp); // Pass to helper method
    }
    catch (Genexcept& Genex) {
        Genex.reportErr();
    }
    // Reset and give output to main window
    this->setCaption(*m_szFilename);
    m_pStatbar->message(trUtf8("File %1 saved").arg(*m_szFilename), 4000);
    this->setDirty(false);  // Set the clean state to allow close
}

//
// Serialize current state to a selected file
//
void Titraqform::saveAs(void)
{
    // First get the selected file name to save to
    QString Filestring = QFileDialog::getSaveFileName(QString::null, QString::null, this);
    if (!Filestring.isEmpty()) {
        *m_szFilename = Filestring;
        saveFile(); // Finish off by calling the save action
    }
    else {
        // User did not select a valid file and push okay button
        m_pStatbar->message(trUtf8("Saving aborted"), 4000);
    }
}

//
// Close current document, displaying in main window
//
void Titraqform::closeEvent(QCloseEvent *pClosit)
{
    int nResult = 0;

    // Check modification state of current data
    if (this->isDirty()) {
        nResult = QMessageBox::information(this, trUtf8(TITRAQ_APPTITLE),
                  trUtf8(TITRAQ_SAVEFIRST), trUtf8("&Save"),
                  trUtf8("&Discard"), trUtf8("Cancel"), 0, 2);

        switch (nResult) {
        case 0: // Save first
            this->saveFile();               // Save changes first
            pClosit->accept();              // then close timesheet
            this->setDirty(false);          // Reset data to clean state
            break;
        case 1: // Don't save first but close
            pClosit->accept();              // Close timesheet
            this->setDirty(false);          // Reset data to clean state
            break;
        case 2: // Don't close
        default:
            pClosit->ignore();
            break;
        }
    }
    else {                                  // Data is already up to date
        pClosit->accept();
        this->setDirty(false);              // Reset data to clean state
    }
}

//
// Edit menu select all entries
//
void Titraqform::selAll(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// Edit a table entry in place, without the usual edit controls
//
void Titraqform::inplaceEdit(int nRow, int nCol, int nButton, const QPoint &Mousepos)
{
    m_pMaintable->setReadOnly(false);
    m_pMaintable->editCell(nRow, nCol);
    m_pMaintable->setEdition(nCol);
    m_pMaintable->setReadOnly(true);
}

//
// Update the edit controls widget sizes
//
void Titraqform::updSizes(int nSection, int nOldsize, int nNewsize)
{
    switch (nSection) {
    case TITRAQ_IDXALLCTRLS:
        m_pDateedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXDATE) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        m_pStarttime->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXSTART) - TITRAQ_SPACING);
        m_pEndtime->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXFINISH) - TITRAQ_SPACING);
        m_pAmount->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXAMOUNT) - TITRAQ_SPACING);
        m_pTasks->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXTASK) - TITRAQ_SPACING);
//        m_pRemark->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXREMARK) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXDATE:
        m_pDateedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXDATE) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    case TITRAQ_IDXSTART:
        m_pStarttime->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXSTART) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXFINISH:
        m_pEndtime->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXFINISH) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXAMOUNT:
        m_pAmount->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXAMOUNT) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXTASK:
        m_pTasks->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXTASK) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXREMARK:
//        m_pRemark->setFixedWidth(nNewsize);
        break;
    default:
        throw Genexcept("Unrecognized main window column header.");
        break;
    }
}

//
//
//
void Titraqform::dataChanged(int nRow, int nCol)
{
    setDirty(); // Mark this timesheet dirty, changes pending save
}

//
// Update the edit controls contents
//
void Titraqform::updEdit(int nRow, int nCol)
{
    QRegExp Shorten("/(\\w+)$");    // For stripping prefix off the current task

    // Field strings to check for validity and process
    QString Textdate(m_pMaintable->text(nRow, TITRAQ_IDXDATE));
    QString Textstart(m_pMaintable->text(nRow, TITRAQ_IDXSTART));
    QString Textfinish(m_pMaintable->text(nRow, TITRAQ_IDXFINISH));
    QString Textamount(m_pMaintable->text(nRow, TITRAQ_IDXAMOUNT));
    QString Texttask(m_pMaintable->text(nRow, TITRAQ_IDXTASK));
    QString Textremark(m_pMaintable->text(nRow, TITRAQ_IDXREMARK));

    // Reset the edition state member
    m_pMaintable->setEdition();

    if (!Textdate.isEmpty())
        m_pDateedit->setDate(QDate::fromString(Textdate, Qt::ISODate));
    else
        m_pDateedit->setDate(*m_pDatezero);

    if (!Textstart.isEmpty())
        m_pStarttime->setTime(QTime::fromString(Textstart, Qt::ISODate));
    else
        m_pStarttime->setTime(QTime::QTime(0, 0));

    if (!Textfinish.isEmpty())
        m_pEndtime->setTime(QTime::fromString(Textfinish, Qt::ISODate));
    else
        m_pEndtime->setTime(QTime::QTime(0, 0));

    m_pAmount->setText(Textamount);

    // Process the task combo box to compress text length
    Texttask.remove(0, Shorten.search(Texttask) + 1); // Strip leading slash
    m_pTasks->setCurrentText(Texttask);

    m_pRemark->setText(Textremark);
}

//
// Update the current date column item
//
void Titraqform::updateDate(const QDate &Dateup)
{
    if (Dateup == *m_pDatezero) // Complete the already nasty hack
        m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXDATE, trUtf8(TITRAQ_DATEZERO));
    else
        m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXDATE, Dateup.toString(Qt::ISODate));
}

//
// Update the current start column item
//
void Titraqform::updateStart(const QTime &Startup)
{
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXSTART, Startup.toString(Qt::ISODate));
}

//
// Update the current finish column item
//
void Titraqform::updateFinish(const QTime &Finishup)
{
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXFINISH, Finishup.toString(Qt::ISODate));
}

//
// Update the current amount column item
//
void Titraqform::updateAmount(const QString &Amountup)
{
    // Ignore incoming data and use widget managed data because its easier
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXAMOUNT, m_pAmount->text());
}

//
// Update the current task column item
//
void Titraqform::updateTask(const QString &Taskup)
{
//    // FIXME: Broken
//    RtTableItem *pTask = NULL;
//    pTask = static_cast<RtTableItem *>(m_pMaintable->item(m_pMaintable->currentRow(), TITRAQ_IDXTASK));
//    pTask->setText(Taskup);

    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXTASK, Taskup);
}

//
// Update the current remark column item
//
void Titraqform::updateRemark(const QString &Remarkup)
{
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXREMARK, Remarkup);
}

//
// Confirm any recent editions on a whole row
//
void Titraqform::confirmEdit(void)
{
    RtTableItem *pTask = NULL;  // Task item is a derived class

    // Conversions from edit control data formats to native tabular format
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXDATE, m_pDateedit->date().toString(Qt::ISODate));
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXSTART, m_pStarttime->time().toString(Qt::ISODate));
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXFINISH, m_pEndtime->time().toString(Qt::ISODate));
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXAMOUNT, m_pAmount->text());

    // Specially handle task fields
    pTask = static_cast<RtTableItem *>(m_pMaintable->item(m_pMaintable->currentRow(), TITRAQ_IDXTASK));
    pTask->setText(m_pTasks->currentText());

    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXREMARK, m_pRemark->text());
}

//
// Edit menu configure preferences
//
void Titraqform::configPrefs(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// View menu normal
//
void Titraqform::normalView(void)
{
    // All view types except normal are disabled until implemention, so
    // this body can remain empty, causing nothing to happen on selection.
}

//
// View menu editing
//
void Titraqform::editingView(void)
{
    // All view types except normal are disabled until implemention, so
    // this body can remain empty, causing nothing to happen on selection.
}

//
// View menu timing
//
void Titraqform::timingView(void)
{
    // All view types except normal are disabled until implemention, so
    // this body can remain empty, causing nothing to happen on selection.
}

//
// View menu show file toolbar
//
void Titraqform::showFilebar(void)
{
    if (m_pFiletools->isVisible()) {
        m_pFiletools->hide();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXFILEBAR), false);
    }
    else {
        m_pFiletools->show();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXFILEBAR), true);
    }
}

//
// View menu show edit toolbar
//
void Titraqform::showEditbar(void)
{
    if (m_pEdittools->isVisible()) {
        m_pEdittools->hide();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXEDITBAR), false);
    }
    else {
        m_pEdittools->show();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXEDITBAR), true);
    }
}

//
// View menu show whats this toolbar
//
void Titraqform::showWhatsbar(void)
{
    if (m_pWhatstools->isVisible()) {
        m_pWhatstools->hide();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXWHATBAR), false);
    }
    else {
        m_pWhatstools->show();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXWHATBAR), true);
    }
}

//
// View menu show line numbers column
//
void Titraqform::showLinecol(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// View menu show GUIDs column
//
void Titraqform::showGuidcol(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// View menu show Dates column
//
void Titraqform::showDatecol(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// View menu show Users column
//
void Titraqform::showUsercol(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// View menu show Tasks column
//
void Titraqform::showTaskcol(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// View menu show Start time column
//
void Titraqform::showStartcol(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// View menu show Finish time column
//
void Titraqform::showFinishcol(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// View menu show Amounts column
//
void Titraqform::showAmountcol(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// View menu show Remarks column
//
void Titraqform::showRemarkcol(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// Get help on Titraq functionality
//
void Titraqform::helpContents(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// Learn more about this program itself
//
void Titraqform::aboutTitraq(void)
{
    QMessageBox *pCwmsg = new QMessageBox("OSSP titraq",
        QObject::trUtf8("OSSP titraq is a time and task-based\n"
               "accounting system that acts as both a\n"
               "work-like punch card and time tracker.\n"
               "Development of titraq is sponsored by\n"
               "Cable & Wireless Deutschland GmbH."),
        QMessageBox::NoIcon, QMessageBox::Ok | QMessageBox::Default,
        QMessageBox::NoButton, QMessageBox::NoButton,
        NULL, "Titraqmessage", true, Qt::WStyle_NormalBorder);

    pCwmsg->setIconPixmap(QPixmap(s_kpcCwlogo_xpm));
    pCwmsg->exec();
}

//
// Learn more about the OSSP
//
void Titraqform::aboutOSSP(void)
{
    QMessageBox *pOsspmsg = new QMessageBox("OSSP titraq",
        QObject::trUtf8("The open source software project (OSSP) is\n"
               "a collective effort aimed at implementing\n"
               "high-quality Unix software components,\n"
               "ranging from networking, multi-threading\n"
               "and algorithmic libraries to networking\n"
               "servers and development tools."),
        QMessageBox::NoIcon, QMessageBox::Ok | QMessageBox::Default,
        QMessageBox::NoButton, QMessageBox::NoButton,
        NULL, "Osspmessage", true, Qt::WStyle_NormalBorder);

    pOsspmsg->setIconPixmap(QPixmap(s_kpcOssplogo_xpm));
    pOsspmsg->exec();
}

//
// Learn more about this program and Qt
//
void Titraqform::aboutQt(void)
{
    QMessageBox::aboutQt(this, "OSSP titraq");
}
