/*
**  L2 - OSSP Logging Library
**  Copyright (c) 2001 The OSSP Project (http://www.ossp.org/)
**  Copyright (c) 2001 Cable & Wireless Deutschland (http://www.cw.com/de/)
**
**  This file is part of OSSP L2, a flexible logging library which
**  can be found at http://www.ossp.org/pkg/l2/.
**
**  Permission to use, copy, modify, and distribute this software for
**  any purpose with or without fee is hereby granted, provided that
**  the above copyright notice and this permission notice appear in all
**  copies.
**
**  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
**  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
**  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
**  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
**  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
**  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
**  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
**  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
**  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
**  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
**  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
**  SUCH DAMAGE.
**
**  l2_channel.c: channel handling
**
*/

#include "l2.h"
#include "l2_p.h"

#include <string.h>

/* create channel */
l2_channel_t *l2_channel_create(l2_handler_t *h)
{
    l2_channel_t *ch;

    /* argument sanity check */
    if (h == NULL)
        return NULL;

    /* allocate channel structure */
    if ((ch = (l2_channel_t *)malloc(sizeof(l2_channel_t))) == NULL)
        return NULL;

    /* initialize channel structure */
    ch->state = L2_CHSTATE_CREATED;
    ch->downstream = NULL;
    memset(&ch->context, 0, sizeof(l2_context_t));
    memcpy(&ch->handler, h, sizeof(l2_handler_t));

    /* (optionally) perform create operation in handler */
    if (ch->handler.create != NULL) {
        if (ch->handler.create(&ch->context, ch) != L2_OK) {
            free(ch);
            return NULL;
        }
    }

    return ch;
}

/* configure channel */
l2_result_t l2_channel_configure(l2_channel_t *ch, const char *fmt, ...)
{
    l2_result_t rv;
    va_list ap;

    /* argument sanity check */
    if (ch == NULL || fmt == NULL)
        return L2_ERR_ARG;

    /* make sure the channel is in state "created" */
    if (ch->state != L2_CHSTATE_CREATED)
        return L2_ERR_USE;

    /* pass operation to handler */
    rv = L2_OK;
    va_start(ap, fmt);
    if (ch->handler.configure != NULL)
        rv = ch->handler.configure(&ch->context, ch, fmt, ap);
    va_end(ap);

    return rv;
}

/* open channel (stack) */
l2_result_t l2_channel_open(l2_channel_t *ch)
{
    l2_result_t rv;

    /* argument sanity check */
    if (ch == NULL)
        return L2_ERR_ARG;

    /* make sure channel is in state "created" */
    if (ch->state != L2_CHSTATE_CREATED)
        return L2_ERR_USE;

    /* skip empty open handlers on channel stack */
    while (ch != NULL && ch->handler.open == NULL) {
        ch->state = L2_CHSTATE_OPENED;
        ch = ch->downstream;
    }
    if (ch == NULL)
        return L2_ERR_USE;

    /* pass operation to handler */
    rv = ch->handler.open(&ch->context, ch);
    if (rv == L2_OK)
        ch->state = L2_CHSTATE_OPENED;

    return rv;
}

/* write to channel (stack) */
l2_result_t l2_channel_write(l2_channel_t *ch, l2_level_t level, const char *buf, size_t bufsize)
{
    l2_result_t rv;

    /* argument sanity check */
    if (ch == NULL || buf == NULL)
        return L2_ERR_ARG;

    /* make sure channel is in state "opened" */
    if (ch->state != L2_CHSTATE_OPENED)
        return L2_ERR_USE;

    /* short circuiting */
    if (bufsize == 0)
        return L2_OK;

    /* walk to next available write handler */
    while (ch != NULL && ch->handler.write == NULL)
        ch = ch->downstream;
    if (ch == NULL)
        return L2_ERR_USE;

    /* pass operation to handler */
    rv = ch->handler.write(&ch->context, ch, level, buf, bufsize);

    return rv;
}

/* flush channel (stack) */
l2_result_t l2_channel_flush(l2_channel_t *ch)
{
    l2_result_t rv;

    /* argument sanity check */
    if (ch == NULL)
        return L2_ERR_ARG;

    /* make sure channel is in state "opened" */
    if (ch->state != L2_CHSTATE_OPENED)
        return L2_ERR_USE;

    /* walk to next available flush handler */
    while (ch != NULL && ch->handler.flush == NULL)
        ch = ch->downstream;
    if (ch == NULL)
        return L2_ERR_USE;

    /* pass operation to handler */
    rv = ch->handler.flush(&ch->context, ch);

    return rv;
}

/* close channel (stack) */
l2_result_t l2_channel_close(l2_channel_t *ch)
{
    l2_result_t rv;

    /* argument sanity check */
    if (ch == NULL)
        return L2_ERR_ARG;
    
    /* make sure channel is in state "opened" */
    if (ch->state != L2_CHSTATE_OPENED)
        return L2_ERR_USE;

    /* walk to next available close handler */
    while (ch != NULL && ch->handler.close == NULL) {
        ch->state = L2_CHSTATE_CREATED;
        ch = ch->downstream;
    }
    if (ch == NULL)
        return L2_ERR_USE;

    /* pass operation to handler */
    rv = ch->handler.close(&ch->context, ch);
    if (rv == L2_OK)
        ch->state = L2_CHSTATE_CREATED;

    return rv;
}

/* destroy channel */
l2_result_t l2_channel_destroy(l2_channel_t *ch)
{
    l2_result_t rv;

    /* argument sanity check */
    if (ch == NULL)
        return L2_ERR_ARG;

    /* make sure channel is in state "opened" */
    if (ch->state == L2_CHSTATE_OPENED)
        if ((rv = l2_channel_close(ch)) != L2_OK)
            return rv;

    /* walk to next available destroy handler */
    while (ch != NULL && ch->handler.destroy == NULL)
        ch = ch->downstream;
    
    /* pass operation to handler */
    if (ch != NULL)
        rv = ch->handler.destroy(&ch->context, ch);
    else
        rv = L2_OK;

    /* free channel structure */
    free(ch);

    return rv;
}

/* stack channel on top of another channel */
l2_result_t l2_channel_stack(l2_channel_t *ch, l2_channel_t *chTop)
{
    /* argument sanity check */
    if (ch == NULL || chTop == NULL)
        return L2_ERR_ARG;

    /* make sure both channels are in state "created" */
    if (   ch->state    != L2_CHSTATE_CREATED
        || chTop->state != L2_CHSTATE_CREATED)
        return L2_ERR_USE;

    /* make sure top channel is a filter channel */
    if (chTop->handler.type != L2_CHANNEL_FILTER)
        return L2_ERR_USE;

    /* stack the channels */
    chTop->downstream = ch;

    return L2_OK;
}

/* return downstream channel */
l2_channel_t *l2_channel_downstream(l2_channel_t *ch)
{
    /* argument sanity check */
    if (ch == NULL)
        return NULL;

    return ch->downstream;
}

/* return channel type */
l2_chtype_t l2_channel_type(l2_channel_t *ch)
{
    /* argument sanity check */
    if (ch == NULL)
        return L2_CHANNEL_FILTER; /* FIXME */

    return ch->handler.type;
}

