/*
**  L2 - OSSP Logging Library
**  Copyright (c) 2001 Cable & Wireless Deutschland GmbH
**
**  This file is part of OSSP L2, a flexible logging library which
**  can be found at http://www.ossp.org/pkg/l2/.
**
**  Permission to use, copy, modify, and distribute this software for
**  any purpose with or without fee is hereby granted, provided that
**  the above copyright notice and this permission notice appear in all
**  copies.
**
**  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
**  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
**  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
**  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
**  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
**  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
**  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
**  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
**  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
**  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
**  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
**  SUCH DAMAGE.
**
**  l2_p.h: private C API
*/

#ifndef __L2_P_H__
#define __L2_P_H__

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "l2.h"
#include "l2_config.h"
#include "l2_ut_pcre.h"
#include "l2_ut_sa.h"

#if defined(HAVE_DMALLOC_H) && defined(WITH_DMALLOC)
#include "dmalloc.h"
#endif

#define L2_MAX_MSGSIZE    4096
#define L2_MAX_CHANNELS    128
#define L2_MAX_FORMATTERS  128

typedef enum {
    L2_CHSTATE_CREATED,
    L2_CHSTATE_OPENED
} l2_chstate_t;

struct l2_channel_st {
    l2_chstate_t  state;
    l2_channel_t *downstream;
    l2_context_t  context;
    l2_handler_t  handler;
};

typedef struct {
    l2_channel_t *ch;
    unsigned int levelmask;
    unsigned int flushmask;
} l2_channel_entry_t;

typedef struct {
    l2_formatter_t cb;
    void *ctx;
    char id;
} l2_formatter_entry_t;

struct l2_stream_st {
    unsigned int          levelmask;
    unsigned int          flushmask;
    char                  message[L2_MAX_MSGSIZE];
    l2_channel_entry_t    channels[L2_MAX_CHANNELS];
    l2_formatter_entry_t  formatters[L2_MAX_FORMATTERS];
};

/* variable argument handling taking care on argument passing conventions */
#define _va_type_recv_char    int
#define _va_type_cast_char    char
#define _va_type_recv_short   int
#define _va_type_cast_short   short
#define _va_type_recv_int     int
#define _va_type_cast_int     int
#define _va_type_recv_long    long
#define _va_type_cast_long    long
#define _va_type_recv_float   double
#define _va_type_cast_float   double
#define _va_type_recv_double  double
#define _va_type_cast_double  double
#define _va_type_recv_charptr void *
#define _va_type_cast_charptr char *
#define _va_type_recv_voidptr void *
#define _va_type_cast_voidptr void *
#define va_get(ap,type) (_va_type_cast_##type)va_arg((ap),_va_type_recv_##type)

struct l2_util_format_st {

    /* the output buffer */
    char *curpos;                        /* start of output buffer (first pos) */
    char *endpos;                        /* end   of output buffer (last pos)  */

    /* callback for flushing the output buffer */
    int (*flush)(
        struct l2_util_format_st *spec   /* this l2_util_format_t specification */
    );

    /* callback for formatting unknown %-constructs */
    void (*format)(
        struct l2_util_format_st *spec,  /* this l2_util_format_t specification */
        char *prefix_char,               /* output arg: prefix character */
        char *pad_char,                  /* output arg: padding character */
        char **s_buf,                    /* output arg: string buffer */
        size_t *s_len,                   /* output arg: string len */
        char *num_buf,                   /* input  arg: temporary buffer */
        int num_buf_size,                /* input  arg: temporary buffer len */
        char *extinfo,                   /* input  arg: extension information */
        char fmt_char,                   /* input  arg: current formatting character */ 
        va_list *ap                      /* in/out arg: variable argument pointer */
    );

    /* arbitrary passed-through application data */
    union { 
        int i;                        
        long l; 
        double d; 
        void *vp; 
    } data[6];                           
};
typedef struct l2_util_format_st l2_util_format_t;

int   l2_util_format     (l2_util_format_t *vbuff, const char *fmt, va_list ap);
int   l2_util_vsprintf   (char *s, size_t n, const char *fmt, va_list ap);
char *l2_util_vasprintf  (const char *fmt, va_list ap);
int   l2_util_sprintf    (char *s, size_t n, const char *fmt, ...);
char *l2_util_asprintf   (const char *fmt, ...);

#endif /* __L2_P_H__ */

