/*
**  L2 - OSSP Logging Library
**  Copyright (c) 2001 The OSSP Project (http://www.ossp.org/)
**  Copyright (c) 2001 Cable & Wireless Deutschland (http://www.cw.com/de/)
**
**  This file is part of OSSP L2, a flexible logging library which
**  can be found at http://www.ossp.org/pkg/l2/.
**
**  Permission to use, copy, modify, and distribute this software for
**  any purpose with or without fee is hereby granted, provided that
**  the above copyright notice and this permission notice appear in all
**  copies.
**
**  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
**  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
**  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
**  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
**  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
**  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
**  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
**  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
**  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
**  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
**  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
**  SUCH DAMAGE.
**
**  l2_ch_buffer.c: buffering channel implementation
*/

#include "l2.h"
#include "l2_p.h"   /* for TRACE() */

#include <string.h>
#include <unistd.h> /* has setitime() in standard unix functions */
#include <signal.h> /* for use of sigaction() and SIGALRM        */

/* declare private channel configuration */
typedef struct {
    char       *buf;
    int         bufpos;
    int         bufsize;
    int         bufinterval;
    struct      sigaction sigalrm;
    struct      itimerval valprev;
    int         levelflush;
    l2_level_t  level;
} l2_ch_buffer_t;

static void catchsignal(int sig, ...)
{
    va_list ap;
    static  l2_channel_t *ch = NULL;

    if (sig == 0) {
        va_start(ap, sig);
        ch = va_arg(ap, l2_channel_t *);
        va_end(ap);
    }
    else if (sig == SIGALRM) {
        TRACE("SIGALRM caught");
        l2_channel_flush(ch);
    }
}

/* create channel */
static l2_result_t hook_create(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_buffer_t *cfg;

    /* allocate private channel configuration */
    if ((cfg = (l2_ch_buffer_t *)malloc(sizeof(l2_ch_buffer_t))) == NULL)
        return L2_ERR_MEM;

    /* initialize configuration with reasonable defaults */
    cfg->buf         = NULL;
    cfg->bufpos      = 0;
    cfg->bufsize     = 4096;
    cfg->bufinterval = 0;
    cfg->levelflush  = 0;
    cfg->level       = L2_LEVEL_NONE;
    memset(&cfg->sigalrm, 0, sizeof(cfg->sigalrm));
    memset(&cfg->valprev, 0, sizeof(cfg->valprev));

    /* link private channel configuration into channel context */
    ctx->vp = cfg;

    return L2_OK;
}

/* configure channel */
static l2_result_t hook_configure(l2_context_t *ctx, l2_channel_t *ch, const char *fmt, va_list ap)
{
    l2_ch_buffer_t *cfg = (l2_ch_buffer_t *)ctx->vp;
    l2_param_t pa[4];
    l2_result_t rv;

    /* feed and call generic parameter parsing engine */
    L2_PARAM_SET(pa[0], size,       INT, &cfg->bufsize);
    L2_PARAM_SET(pa[1], interval,   INT, &cfg->bufinterval);
    L2_PARAM_SET(pa[2], levelflush, INT, &cfg->levelflush);
    L2_PARAM_END(pa[3]);
    rv = l2_util_setparams(pa, fmt, ap);
    if (cfg->bufsize < 0) 
        return L2_ERR_ARG;

    return rv;
}

/************************************************************
 * set_alarm: Helper method to hook_open                    *
 *   Sets the VIRTUAL timer to preconfigured value in cfg   *
 ************************************************************/
static int set_alarm(l2_ch_buffer_t *cfg)
{
    struct itimerval valnew;

    /* initialize auto vars before using them */
    memset(&valnew, 0, sizeof(valnew));

    valnew.it_interval.tv_sec = cfg->bufinterval;
    valnew.it_interval.tv_usec = 0;
    valnew.it_value.tv_sec = cfg->bufinterval;
    valnew.it_value.tv_usec = 0;
    return setitimer(ITIMER_REAL, &valnew, &cfg->valprev); /* set flush timer */
}

/************************************************************
 * reset_alarm: Helper method to hook_flush                 *
 *   Resets the VIRTUAL timer to preconfigured value in cfg *
 ************************************************************/
static int reset_alarm(l2_ch_buffer_t *cfg)
{
    struct itimerval valnew;

    /* initialize auto vars before using them */
    memset(&valnew, 0, sizeof(valnew));

    valnew.it_interval.tv_sec = cfg->bufinterval;
    valnew.it_interval.tv_usec = 0;
    valnew.it_value.tv_sec = cfg->bufinterval;
    valnew.it_value.tv_usec = 0;
    return setitimer(ITIMER_REAL, &valnew, 0); /* reset flush timer */
}

/* open channel */
static l2_result_t hook_open(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_buffer_t *cfg = (l2_ch_buffer_t *)ctx->vp;
    l2_channel_t *downstream = l2_channel_downstream(ch);
    struct sigaction locact;
    l2_result_t rv;

    if (cfg->bufinterval != 0) {
        /* initialize auto vars before using them */
        memset(&locact, 0, sizeof(locact));

        locact.sa_handler = (void(*)())catchsignal;
        sigemptyset(&locact.sa_mask);
        locact.sa_flags = 0;

        catchsignal(0, ch);
        /* save old signal context before replacing with our own */
        if (sigaction(SIGALRM, &locact, &cfg->sigalrm) < 0)
            return L2_ERR_SYS;

        if (set_alarm(cfg))
            return L2_ERR_SYS;
    }

    /* open channel buffer */
    if (cfg->bufsize > 0) {
        if ((cfg->buf = malloc(cfg->bufsize)) == NULL)
            return L2_ERR_MEM;
        cfg->bufpos = 0;
    }

    /* optionally open downstream channel, too */
    if ((rv = l2_channel_open(downstream)) != L2_OK)
        return rv;

    return L2_OK;
}

/* write to channel */
static l2_result_t hook_write(l2_context_t *ctx, l2_channel_t *ch,
                              l2_level_t level, const char *buf, size_t buf_size)
{
    l2_ch_buffer_t *cfg = (l2_ch_buffer_t *)ctx->vp;
    l2_channel_t *downstream = l2_channel_downstream(ch);
    l2_result_t rv;

    if (buf_size > (cfg->bufsize - cfg->bufpos)) {
        /* flush buffer if necessary */
        if (cfg->bufpos > 0) {
            if ((rv = l2_channel_write(downstream, cfg->level, cfg->buf, cfg->bufpos)) != L2_OK)
                return rv;
            cfg->bufpos = 0;
            cfg->level  = L2_LEVEL_NONE;
        }
        /* pass through immediately to downstream if still too large */
        if (buf_size > cfg->bufsize)
            return l2_channel_write(downstream, level, buf, buf_size);
    }

    /* flush if level of incoming message differs from those already in buffer */
    if (   (cfg->levelflush)             /* if different levels force a flush    */
        && (cfg->bufpos > 0)             /* and there is something in the buffer */
        && (cfg->level != L2_LEVEL_NONE) /* and a remembered level is known      */
        && (level != cfg->level)         /* and the levels really differ         */) {
        if ((rv = l2_channel_write(downstream, cfg->level, cfg->buf, cfg->bufpos)) != L2_OK)
            return rv;
        cfg->bufpos = 0;
        cfg->level  = L2_LEVEL_NONE;
    }

    /* finally write incoming message to channel buffer */
    memcpy(cfg->buf+cfg->bufpos, buf, buf_size);
    cfg->bufpos += buf_size;
    cfg->level = level;

    return L2_OK;
}

/* flush channel */
static l2_result_t hook_flush(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_buffer_t *cfg = (l2_ch_buffer_t *)ctx->vp;
    l2_channel_t *downstream = l2_channel_downstream(ch);
    l2_result_t rv;

    /* write the buffer contents downstream */
    if (cfg->bufpos > 0) {
        if ((rv = l2_channel_write(downstream, cfg->level, cfg->buf, cfg->bufpos)) != L2_OK)
            return rv;
        cfg->bufpos = 0;
        cfg->level  = L2_LEVEL_NONE; /* reset this->context->level */
    }

    /* reset the flush alarm timer to synchronize the buffer */
    if (cfg->bufinterval != 0)
        if (reset_alarm(cfg))
            return L2_ERR_SYS;

    /* optionally flush downstream channel, too */
    if ((rv = l2_channel_flush(downstream)) != L2_OK)
        return rv;

    return L2_OK;
}

/* close channel */
static l2_result_t hook_close(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_buffer_t *cfg = (l2_ch_buffer_t *)ctx->vp;
    l2_channel_t *downstream = l2_channel_downstream(ch);
    l2_result_t rv;

    if (cfg->bufinterval != 0) {
        setitimer(ITIMER_REAL, &cfg->valprev, 0); /* restore timer values */
        /* restore previous signal context if previously saved & replaced    */
        if (&cfg->sigalrm.sa_handler)
            if (sigaction(SIGALRM, &cfg->sigalrm, 0) < 0)
                rv = L2_ERR_SYS;
    }

    /* write pending data before closing down */
    if (cfg->bufpos > 0) {
        if ((rv = l2_channel_write(downstream, cfg->level, cfg->buf, cfg->bufpos)) != L2_OK)
            return rv;
        cfg->bufpos = 0;
        cfg->level  = L2_LEVEL_NONE; /* reset this->context->level */
    }

    /* optionally close downstream channel, too */
    if ((rv = l2_channel_close(downstream)) != L2_OK)
        return rv;

    /* close channel buffer */
    if (cfg->buf != NULL) {
        free(cfg->buf);
        cfg->buf = NULL;
    }

    return L2_OK;
}

/* destroy channel */
static l2_result_t hook_destroy(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_buffer_t *cfg = (l2_ch_buffer_t *)ctx->vp;
    l2_channel_t *downstream = l2_channel_downstream(ch);
    l2_result_t rv;

    /* if not already closed, close channel buffer now */
    if (cfg->buf != NULL) {
        free(cfg->buf);
        cfg->buf = NULL;
    }

    /* destroy channel configuration */
    free(cfg);

    /* optionally destroy downstream channel, too */
    if ((rv = l2_channel_destroy(downstream)) != L2_OK)
        return rv;

    return L2_OK;
}

/* exported channel handler structure */
l2_handler_t l2_handler_buffer = {
    L2_CHANNEL_FILTER,
    hook_create,
    hook_configure,
    hook_open,
    hook_write,
    hook_flush,
    hook_close,
    hook_destroy
};

